<?php
require_once '../config/config.php';

if (!isAdmin()) {
    redirect('../dashboard.php');
}

$db = Database::getInstance()->getConnection();

// Handle actions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    
    if ($action === 'create_desa') {
        $nama_desa = sanitize($_POST['nama_desa'] ?? '');
        $kode_desa = sanitize($_POST['kode_desa'] ?? '');
        
        if (empty($nama_desa)) {
            $_SESSION['error'] = 'Nama desa harus diisi';
        } else {
            $stmt = $db->prepare("INSERT INTO desa (nama_desa, kode_desa) VALUES (?, ?)");
            if ($stmt->execute([$nama_desa, $kode_desa])) {
                $_SESSION['success'] = 'Desa berhasil ditambahkan';
            } else {
                $_SESSION['error'] = 'Gagal menambahkan desa';
            }
        }
    } elseif ($action === 'create_dusun') {
        $desa_id = intval($_POST['desa_id'] ?? 0);
        $nama_dusun = sanitize($_POST['nama_dusun'] ?? '');
        $rt = sanitize($_POST['rt'] ?? '');
        $rw = sanitize($_POST['rw'] ?? '');
        $alamat = sanitize($_POST['alamat'] ?? '');
        $latitude = floatval($_POST['latitude'] ?? 0);
        $longitude = floatval($_POST['longitude'] ?? 0);
        $api_bmkg_url = sanitize($_POST['api_bmkg_url'] ?? '');
        
        if (empty($nama_dusun) || $desa_id == 0) {
            $_SESSION['error'] = 'Data wajib harus diisi';
        } else {
            $stmt = $db->prepare("INSERT INTO dusun (desa_id, nama_dusun, rt, rw, alamat, latitude, longitude, api_bmkg_url) VALUES (?, ?, ?, ?, ?, ?, ?, ?)");
            if ($stmt->execute([$desa_id, $nama_dusun, $rt, $rw, $alamat, $latitude, $longitude, $api_bmkg_url])) {
                $_SESSION['success'] = 'Dusun berhasil ditambahkan';
            } else {
                $_SESSION['error'] = 'Gagal menambahkan dusun';
            }
        }
    } elseif ($action === 'update_dusun') {
        $dusun_id = intval($_POST['dusun_id'] ?? 0);
        $nama_dusun = sanitize($_POST['nama_dusun'] ?? '');
        $rt = sanitize($_POST['rt'] ?? '');
        $rw = sanitize($_POST['rw'] ?? '');
        $alamat = sanitize($_POST['alamat'] ?? '');
        $latitude = floatval($_POST['latitude'] ?? 0);
        $longitude = floatval($_POST['longitude'] ?? 0);
        $api_bmkg_url = sanitize($_POST['api_bmkg_url'] ?? '');
        
        $stmt = $db->prepare("UPDATE dusun SET nama_dusun = ?, rt = ?, rw = ?, alamat = ?, latitude = ?, longitude = ?, api_bmkg_url = ? WHERE id = ?");
        if ($stmt->execute([$nama_dusun, $rt, $rw, $alamat, $latitude, $longitude, $api_bmkg_url, $dusun_id])) {
            $_SESSION['success'] = 'Dusun berhasil diupdate';
        } else {
            $_SESSION['error'] = 'Gagal update dusun';
        }
    } elseif ($action === 'delete_dusun') {
        $dusun_id = intval($_POST['dusun_id'] ?? 0);
        $stmt = $db->prepare("DELETE FROM dusun WHERE id = ?");
        if ($stmt->execute([$dusun_id])) {
            $_SESSION['success'] = 'Dusun berhasil dihapus';
        } else {
            $_SESSION['error'] = 'Gagal menghapus dusun';
        }
    }
    
    redirect('desa.php');
}

// Get all desa
$stmt = $db->query("SELECT * FROM desa ORDER BY nama_desa");
$desa_list = $stmt->fetchAll();

// Get all dusun with desa info - only dusun that have valid desa_id
$stmt = $db->query("SELECT d.*, des.nama_desa, des.id as desa_id_real 
                     FROM dusun d 
                     INNER JOIN desa des ON d.desa_id = des.id 
                     WHERE d.desa_id IS NOT NULL AND d.desa_id > 0
                     ORDER BY des.nama_desa, d.nama_dusun");
$dusun_list = $stmt->fetchAll();

// Check for orphaned dusun (dusun without valid desa)
$stmt = $db->query("SELECT COUNT(*) FROM dusun WHERE desa_id IS NULL OR desa_id NOT IN (SELECT id FROM desa)");
$orphaned_count = $stmt->fetchColumn();
if ($orphaned_count > 0) {
    $_SESSION['warning'] = "Peringatan: Ada $orphaned_count dusun yang tidak terhubung ke desa. Silakan perbaiki data.";
}

?>
<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Kelola Desa & Dusun - <?php echo APP_NAME; ?></title>
    <link rel="stylesheet" href="../assets/css/style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <link rel="stylesheet" href="https://unpkg.com/leaflet@1.9.4/dist/leaflet.css" />
    <script src="https://unpkg.com/leaflet@1.9.4/dist/leaflet.js"></script>
</head>
<body>
    <aside class="sidebar">
        <div class="sidebar-header">
            <h2><i class="fas fa-tint"></i> <?php echo APP_NAME; ?></h2>
        </div>
        <nav class="sidebar-nav">
            <a href="../dashboard.php" class="nav-item">
                <i class="fas fa-home"></i>
                <span>Dashboard</span>
            </a>
            <a href="users.php" class="nav-item">
                <i class="fas fa-users"></i>
                <span>Kelola Pengguna</span>
            </a>
            <a href="desa.php" class="nav-item active">
                <i class="fas fa-building"></i>
                <span>Kelola Desa</span>
            </a>
            <a href="menu_icons.php" class="nav-item">
                <i class="fas fa-palette"></i>
                <span>Kelola Menu</span>
            </a>
            <a href="../auth/logout.php" class="nav-item">
                <i class="fas fa-sign-out-alt"></i>
                <span>Logout</span>
            </a>
        </nav>
    </aside>
    
    <main class="main-content">
        <header class="content-header">
            <div class="header-left">
                <a href="../dashboard.php" class="btn btn-secondary" style="margin-bottom: 10px; display: inline-flex; align-items: center; gap: 8px; text-decoration: none;">
                    <i class="fas fa-arrow-left"></i> Kembali ke Dashboard
                </a>
                <h1><i class="fas fa-building"></i> Kelola Desa & Dusun</h1>
                <p>Manajemen data desa dan dusun</p>
            </div>
            <div>
                <button class="btn btn-primary" onclick="showDesaModal()">
                    <i class="fas fa-plus"></i> Tambah Desa
                </button>
                <button class="btn btn-success" onclick="showDusunModal()">
                    <i class="fas fa-plus"></i> Tambah Dusun
                </button>
            </div>
        </header>

        <?php if (isset($_SESSION['success'])): ?>
        <div class="alert alert-success">
            <i class="fas fa-check-circle"></i> <?php echo $_SESSION['success']; unset($_SESSION['success']); ?>
        </div>
        <?php endif; ?>

        <?php if (isset($_SESSION['error'])): ?>
        <div class="alert alert-danger">
            <i class="fas fa-exclamation-circle"></i> <?php echo $_SESSION['error']; unset($_SESSION['error']); ?>
        </div>
        <?php endif; ?>

        <?php if (isset($_SESSION['warning'])): ?>
        <div class="alert alert-warning" style="background-color: #fff3cd; border-color: #ffc107; color: #856404;">
            <i class="fas fa-exclamation-triangle"></i> <?php echo $_SESSION['warning']; unset($_SESSION['warning']); ?>
        </div>
        <?php endif; ?>

        <!-- Peta Interaktif dengan Pencarian -->
        <div class="content-section">
            <h2><i class="fas fa-map-marked-alt"></i> Peta Desa & Dusun</h2>
            <div style="margin-bottom: 15px;">
                <div style="display: flex; gap: 10px; flex-wrap: wrap; align-items: center;">
                    <div style="flex: 1; min-width: 250px;">
                        <input type="text" id="searchInput" placeholder="Cari desa atau dusun..." style="width: 100%; padding: 10px; border: 1px solid #ddd; border-radius: 5px; font-size: 14px;">
                    </div>
                    <button class="btn btn-primary" onclick="searchLocation()" style="white-space: nowrap;">
                        <i class="fas fa-search"></i> Cari
                    </button>
                    <button class="btn btn-secondary" onclick="resetMap()" style="white-space: nowrap;">
                        <i class="fas fa-redo"></i> Reset
                    </button>
                </div>
                <div id="searchResults" style="margin-top: 10px; max-height: 200px; overflow-y: auto; display: none; border: 1px solid #ddd; border-radius: 5px; background: white;">
                    <!-- Results will be populated here -->
                </div>
            </div>
            <div id="mainMap" style="height: 500px; border: 1px solid #ddd; border-radius: 5px;"></div>
        </div>

        <!-- Desa List -->
        <div class="content-section">
            <h2>Daftar Desa</h2>
            <div class="alert alert-info" style="background-color: #d1ecf1; border-color: #bee5eb; color: #0c5460; padding: 12px; border-radius: 5px; margin-bottom: 15px;">
                <i class="fas fa-info-circle"></i> <strong>Info:</strong> Pastikan setiap dusun terhubung ke desa yang benar. Jika ada dusun yang muncul tanpa desa, silakan edit dusun tersebut dan pilih desa yang sesuai.
            </div>
            <div class="table-responsive">
                <table class="data-table">
                    <thead>
                        <tr>
                            <th>Kode Desa</th>
                            <th>Nama Desa</th>
                            <th>Jumlah Dusun</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($desa_list as $desa): 
                            $stmt = $db->prepare("SELECT COUNT(*) FROM dusun WHERE desa_id = ?");
                            $stmt->execute([$desa['id']]);
                            $jumlah_dusun = $stmt->fetchColumn();
                        ?>
                        <tr>
                            <td><?php echo htmlspecialchars($desa['kode_desa'] ?? '-'); ?></td>
                            <td><?php echo htmlspecialchars($desa['nama_desa']); ?></td>
                            <td><?php echo $jumlah_dusun; ?></td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </div>

        <!-- Dusun List -->
        <div class="content-section">
            <h2>Daftar Dusun</h2>
            <div class="table-responsive">
                <table class="data-table">
                    <thead>
                        <tr>
                            <th>No</th>
                            <th>Nama Dusun</th>
                            <th>Desa</th>
                            <th>RT/RW</th>
                            <th>Alamat</th>
                            <th>Koordinat</th>
                            <th>API BMKG</th>
                            <th>Aksi</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php $no = 1; foreach ($dusun_list as $dusun): ?>
                        <tr>
                            <td><?php echo $no++; ?></td>
                            <td><strong><?php echo htmlspecialchars($dusun['nama_dusun'] ?? '-'); ?></strong></td>
                            <td><strong style="color: #2ecc71;"><?php echo htmlspecialchars($dusun['nama_desa'] ?? '-'); ?></strong></td>
                            <td><?php echo htmlspecialchars($dusun['rt'] ?? '-'); ?>/<?php echo htmlspecialchars($dusun['rw'] ?? '-'); ?></td>
                            <td><?php echo htmlspecialchars($dusun['alamat'] ?? '-'); ?></td>
                            <td>
                                <?php if ($dusun['latitude'] && $dusun['longitude']): ?>
                                <?php echo number_format($dusun['latitude'], 6); ?>, <?php echo number_format($dusun['longitude'], 6); ?>
                                <?php else: ?>
                                <span class="badge badge-warning">Belum diatur</span>
                                <?php endif; ?>
                            </td>
                            <td>
                                <?php if (!empty($dusun['api_bmkg_url'])): ?>
                                <a href="<?php echo htmlspecialchars($dusun['api_bmkg_url']); ?>" target="_blank" style="color: #3498db; text-decoration: none; word-break: break-all;">
                                    <i class="fas fa-link"></i> Link
                                </a>
                                <?php else: ?>
                                <span class="badge badge-warning">Belum diatur</span>
                                <?php endif; ?>
                            </td>
                            <td>
                                <button class="btn btn-sm btn-primary" onclick="editDusun(<?php echo htmlspecialchars(json_encode($dusun), ENT_QUOTES, 'UTF-8'); ?>)">
                                    <i class="fas fa-edit"></i>
                                </button>
                                <button class="btn btn-sm btn-danger" onclick="deleteDusun(<?php echo $dusun['id']; ?>)">
                                    <i class="fas fa-trash"></i>
                                </button>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </main>

    <!-- Desa Modal -->
    <div id="desaModal" class="modal" style="display:none;">
        <div class="modal-content">
            <span class="close" onclick="document.getElementById('desaModal').style.display='none'">&times;</span>
            <h2>Tambah Desa</h2>
            <form method="POST" class="form-container">
                <input type="hidden" name="action" value="create_desa">
                <div class="form-group">
                    <label>Nama Desa</label>
                    <input type="text" name="nama_desa" required>
                </div>
                <div class="form-group">
                    <label>Kode Desa</label>
                    <input type="text" name="kode_desa">
                </div>
                <button type="submit" class="btn btn-primary btn-block">Simpan</button>
            </form>
        </div>
    </div>

    <!-- Dusun Modal -->
    <div id="dusunModal" class="modal" style="display:none;">
        <div class="modal-content">
            <span class="close" onclick="document.getElementById('dusunModal').style.display='none'">&times;</span>
            <h2 id="dusunModalTitle">Tambah Dusun</h2>
            <form method="POST" id="dusunForm" class="form-container">
                <input type="hidden" name="action" value="create_dusun" id="dusunAction">
                <input type="hidden" name="dusun_id" id="dusun_id">
                <div class="form-group">
                    <label>Desa</label>
                    <select name="desa_id" id="desa_id" required>
                        <option value="">Pilih Desa</option>
                        <?php foreach ($desa_list as $desa): ?>
                        <option value="<?php echo $desa['id']; ?>"><?php echo htmlspecialchars($desa['nama_desa']); ?></option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <div class="form-group">
                    <label>Nama Dusun</label>
                    <input type="text" name="nama_dusun" id="nama_dusun" required>
                </div>
                <div class="form-row">
                    <div class="form-group">
                        <label>RT</label>
                        <input type="text" name="rt" id="rt">
                    </div>
                    <div class="form-group">
                        <label>RW</label>
                        <input type="text" name="rw" id="rw">
                    </div>
                </div>
                <div class="form-group">
                    <label>Alamat</label>
                    <textarea name="alamat" id="alamat" rows="3"></textarea>
                </div>
                <div class="form-group">
                    <label>Pilih Lokasi di Peta</label>
                    <div id="map" style="height: 300px; margin-top: 10px;"></div>
                    <small>Klik di peta untuk menentukan koordinat</small>
                </div>
                <div class="form-row">
                    <div class="form-group">
                        <label>Latitude</label>
                        <input type="number" step="any" name="latitude" id="latitude" readonly>
                    </div>
                    <div class="form-group">
                        <label>Longitude</label>
                        <input type="number" step="any" name="longitude" id="longitude" readonly>
                    </div>
                </div>
                <div class="form-group">
                    <label><i class="fas fa-cloud-sun-rain"></i> Link API BMKG</label>
                    <input type="url" name="api_bmkg_url" id="api_bmkg_url" placeholder="https://data.bmkg.go.id/api/..." value="">
                    <small style="color: #666; display: block; margin-top: 5px;">
                        <i class="fas fa-info-circle"></i> Masukkan link API BMKG untuk dusun ini (opsional)
                    </small>
                </div>
                <button type="submit" class="btn btn-primary btn-block">Simpan</button>
            </form>
        </div>
    </div>

    <script src="../assets/js/main.js"></script>
    <script>
    let map, marker;
    let mainMap, mainMarkers = [];
    let allDusun = <?php echo json_encode($dusun_list); ?>;
    let allDesa = <?php echo json_encode($desa_list); ?>;
    
    // Initialize main map
    document.addEventListener('DOMContentLoaded', function() {
        mainMap = L.map('mainMap').setView([-7.7956, 110.3695], 11);
        L.tileLayer('https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png', {
            attribution: '© OpenStreetMap contributors'
        }).addTo(mainMap);
        
        // Add markers for all dusun
        allDusun.forEach(function(dusun) {
            if (dusun.latitude && dusun.longitude) {
                const popupContent = `
                    <div style="min-width: 200px;">
                        <strong>${dusun.nama_dusun}</strong><br>
                        <small>Desa: ${dusun.nama_desa}</small><br>
                        ${dusun.rt && dusun.rw ? `<small>RT/RW: ${dusun.rt}/${dusun.rw}</small><br>` : ''}
                        ${dusun.alamat ? `<small>${dusun.alamat}</small><br>` : ''}
                        <button class="btn btn-sm btn-primary" onclick="editDusunFromMap(${dusun.id})" style="margin-top: 5px;">
                            <i class="fas fa-edit"></i> Edit
                        </button>
                    </div>
                `;
                const m = L.marker([parseFloat(dusun.latitude), parseFloat(dusun.longitude)])
                    .addTo(mainMap)
                    .bindPopup(popupContent);
                mainMarkers.push(m);
            }
        });
        
        // Search functionality
        document.getElementById('searchInput').addEventListener('keyup', function(e) {
            if (e.key === 'Enter') {
                searchLocation();
            } else {
                filterSearch();
            }
        });
    });
    
    function filterSearch() {
        const query = document.getElementById('searchInput').value.toLowerCase().trim();
        const resultsDiv = document.getElementById('searchResults');
        
        if (query.length < 2) {
            resultsDiv.style.display = 'none';
            return;
        }
        
        const results = [];
        
        // Search in dusun
        allDusun.forEach(function(dusun) {
            if (dusun.nama_dusun.toLowerCase().includes(query) || 
                dusun.nama_desa.toLowerCase().includes(query) ||
                (dusun.alamat && dusun.alamat.toLowerCase().includes(query))) {
                results.push({
                    type: 'dusun',
                    name: dusun.nama_dusun,
                    desa: dusun.nama_desa,
                    data: dusun
                });
            }
        });
        
        // Search in desa
        allDesa.forEach(function(desa) {
            if (desa.nama_desa.toLowerCase().includes(query)) {
                // Find dusun in this desa
                const dusunInDesa = allDusun.filter(d => d.desa_id == desa.id);
                results.push({
                    type: 'desa',
                    name: desa.nama_desa,
                    count: dusunInDesa.length,
                    data: desa,
                    dusun: dusunInDesa
                });
            }
        });
        
        if (results.length > 0) {
            let html = '<div style="padding: 10px;">';
            results.slice(0, 10).forEach(function(result) {
                if (result.type === 'dusun') {
                    const namaDusun = result.data.nama_dusun.replace(/'/g, "\\'");
                    const namaDesa = result.data.nama_desa.replace(/'/g, "\\'");
                    html += `
                        <div onclick="goToLocation(${result.data.latitude}, ${result.data.longitude}, '${namaDusun}', '${namaDesa}')" 
                             style="padding: 8px; cursor: pointer; border-bottom: 1px solid #eee; hover:background: #f5f5f5;" 
                             onmouseover="this.style.background='#f5f5f5'" 
                             onmouseout="this.style.background='white'">
                            <strong><i class="fas fa-map-marker-alt"></i> ${result.name}</strong><br>
                            <small style="color: #666;">Desa: ${result.desa}</small>
                        </div>
                    `;
                } else {
                    const namaDesa = result.data.nama_desa.replace(/'/g, "\\'");
                    html += `
                        <div onclick="goToDesa(${result.data.id}, '${namaDesa}')" 
                             style="padding: 8px; cursor: pointer; border-bottom: 1px solid #eee;" 
                             onmouseover="this.style.background='#f5f5f5'" 
                             onmouseout="this.style.background='white'">
                            <strong><i class="fas fa-building"></i> ${result.name}</strong><br>
                            <small style="color: #666;">${result.count} dusun</small>
                        </div>
                    `;
                }
            });
            if (results.length > 10) {
                html += `<div style="padding: 8px; color: #666; font-style: italic;">... dan ${results.length - 10} hasil lainnya</div>`;
            }
            html += '</div>';
            resultsDiv.innerHTML = html;
            resultsDiv.style.display = 'block';
        } else {
            resultsDiv.innerHTML = '<div style="padding: 10px; color: #666;">Tidak ada hasil ditemukan</div>';
            resultsDiv.style.display = 'block';
        }
    }
    
    function searchLocation() {
        const query = document.getElementById('searchInput').value.toLowerCase().trim();
        if (!query) return;
        
        // Find exact match first
        const dusunMatch = allDusun.find(d => 
            d.nama_dusun.toLowerCase() === query || 
            d.nama_dusun.toLowerCase().includes(query)
        );
        
        if (dusunMatch && dusunMatch.latitude && dusunMatch.longitude) {
            goToLocation(dusunMatch.latitude, dusunMatch.longitude, dusunMatch.nama_dusun, dusunMatch.nama_desa);
            return;
        }
        
        // Find desa match
        const desaMatch = allDesa.find(d => d.nama_desa.toLowerCase() === query || d.nama_desa.toLowerCase().includes(query));
        if (desaMatch) {
            goToDesa(desaMatch.id, desaMatch.nama_desa);
            return;
        }
        
        // Show filtered results
        filterSearch();
    }
    
    function goToLocation(lat, lng, namaDusun, namaDesa) {
        if (!lat || !lng) return;
        
        mainMap.setView([parseFloat(lat), parseFloat(lng)], 15);
        
        // Highlight the marker
        mainMarkers.forEach(function(m) {
            if (Math.abs(m.getLatLng().lat - parseFloat(lat)) < 0.0001 && 
                Math.abs(m.getLatLng().lng - parseFloat(lng)) < 0.0001) {
                m.openPopup();
            }
        });
        
        document.getElementById('searchResults').style.display = 'none';
        document.getElementById('searchInput').value = namaDusun;
    }
    
    function goToDesa(desaId, namaDesa) {
        const dusunInDesa = allDusun.filter(d => d.desa_id == desaId && d.latitude && d.longitude);
        
        if (dusunInDesa.length > 0) {
            // Calculate center of all dusun in this desa
            let totalLat = 0, totalLng = 0, count = 0;
            dusunInDesa.forEach(function(d) {
                totalLat += parseFloat(d.latitude);
                totalLng += parseFloat(d.longitude);
                count++;
            });
            
            const centerLat = totalLat / count;
            const centerLng = totalLng / count;
            
            mainMap.setView([centerLat, centerLng], 12);
            
            // Open popup for first dusun
            setTimeout(function() {
                const firstDusun = dusunInDesa[0];
                mainMarkers.forEach(function(m) {
                    if (Math.abs(m.getLatLng().lat - parseFloat(firstDusun.latitude)) < 0.0001 && 
                        Math.abs(m.getLatLng().lng - parseFloat(firstDusun.longitude)) < 0.0001) {
                        m.openPopup();
                    }
                });
            }, 500);
        } else {
            // If no dusun with coordinates, just show message
            alert('Desa ' + namaDesa + ' belum memiliki dusun dengan koordinat GPS');
        }
        
        document.getElementById('searchResults').style.display = 'none';
        document.getElementById('searchInput').value = namaDesa;
    }
    
    function resetMap() {
        mainMap.setView([-7.7956, 110.3695], 11);
        document.getElementById('searchInput').value = '';
        document.getElementById('searchResults').style.display = 'none';
        mainMap.closePopup();
    }
    
    function showDesaModal() {
        document.getElementById('desaModal').style.display = 'block';
    }
    
    function showDusunModal() {
        document.getElementById('dusunModal').style.display = 'block';
        document.getElementById('dusunForm').reset();
        document.getElementById('dusun_id').value = '';
        document.getElementById('dusunModalTitle').textContent = 'Tambah Dusun';
        document.getElementById('dusunAction').value = 'create_dusun';
        
        // Initialize map
        setTimeout(() => {
            if (!map) {
                map = L.map('map').setView([-7.7956, 110.3695], 13);
                L.tileLayer('https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png', {
                    attribution: '© OpenStreetMap contributors'
                }).addTo(map);
                
                map.on('click', function(e) {
                    const lat = e.latlng.lat;
                    const lng = e.latlng.lng;
                    document.getElementById('latitude').value = lat;
                    document.getElementById('longitude').value = lng;
                    
                    if (marker) {
                        map.removeLayer(marker);
                    }
                    marker = L.marker([lat, lng]).addTo(map);
                });
            }
        }, 100);
    }
    
    function editDusun(dusun) {
        document.getElementById('dusunModal').style.display = 'block';
        document.getElementById('dusunModalTitle').textContent = 'Edit Dusun';
        document.getElementById('dusunAction').value = 'update_dusun';
        document.getElementById('dusun_id').value = dusun.id;
        document.getElementById('desa_id').value = dusun.desa_id;
        document.getElementById('nama_dusun').value = dusun.nama_dusun;
        document.getElementById('rt').value = dusun.rt || '';
        document.getElementById('rw').value = dusun.rw || '';
        document.getElementById('alamat').value = dusun.alamat || '';
        document.getElementById('latitude').value = dusun.latitude || '';
        document.getElementById('longitude').value = dusun.longitude || '';
        document.getElementById('api_bmkg_url').value = dusun.api_bmkg_url || '';
        
        if (dusun.latitude && dusun.longitude) {
            setTimeout(() => {
                if (!map) {
                    map = L.map('map').setView([dusun.latitude, dusun.longitude], 13);
                    L.tileLayer('https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png').addTo(map);
                    marker = L.marker([dusun.latitude, dusun.longitude]).addTo(map);
                } else {
                    map.setView([dusun.latitude, dusun.longitude], 13);
                    if (marker) map.removeLayer(marker);
                    marker = L.marker([dusun.latitude, dusun.longitude]).addTo(map);
                }
                
                map.on('click', function(e) {
                    const lat = e.latlng.lat;
                    const lng = e.latlng.lng;
                    document.getElementById('latitude').value = lat;
                    document.getElementById('longitude').value = lng;
                    if (marker) map.removeLayer(marker);
                    marker = L.marker([lat, lng]).addTo(map);
                });
            }, 100);
        }
    }
    
    function editDusunFromMap(dusunId) {
        const dusun = allDusun.find(d => d.id == dusunId);
        if (dusun) {
            editDusun(dusun);
        }
    }
    
    function deleteDusun(id) {
        if (confirm('Hapus dusun ini?')) {
            const form = document.createElement('form');
            form.method = 'POST';
            form.innerHTML = `
                <input type="hidden" name="action" value="delete_dusun">
                <input type="hidden" name="dusun_id" value="${id}">
            `;
            document.body.appendChild(form);
            form.submit();
        }
    }
    
    // Close modal when clicking outside
    window.onclick = function(event) {
        const modals = document.querySelectorAll('.modal');
        modals.forEach(modal => {
            if (event.target == modal) {
                modal.style.display = 'none';
            }
        });
    }
    </script>
    <style>
    .modal {
        display: none;
        position: fixed;
        z-index: 1000;
        left: 0;
        top: 0;
        width: 100%;
        height: 100%;
        background: rgba(0,0,0,0.5);
        overflow-y: auto;
    }
    
    .modal-content {
        background: white;
        margin: 3% auto;
        padding: 30px;
        border-radius: 15px;
        width: 90%;
        max-width: 800px;
    }
    
    .close {
        float: right;
        font-size: 28px;
        font-weight: bold;
        cursor: pointer;
        color: #999;
    }
    
    .close:hover {
        color: #333;
    }
    </style>
</body>
</html>

